const fs = require('fs');
const path = require('path');
const logger = require('./logger');

let ffprobePath = null;

function initializeFFprobePath() {
  try {
    const ffprobeInstaller = require('@ffprobe-installer/ffprobe');
    ffprobePath = ffprobeInstaller.path;

    if (ffprobePath && ffprobePath.includes('app.asar')) {
      const unpackedPath = ffprobePath.replace('app.asar', 'app.asar.unpacked');
      if (fs.existsSync(unpackedPath)) {
        ffprobePath = unpackedPath;
      }
    }

    logger.log('FFprobe found at:', ffprobePath);
  } catch (error) {
    logger.log('@ffprobe-installer/ffprobe not available, will try to find ffprobe manually');
  }
}

initializeFFprobePath();

function setFFprobePath(manualPath) {
  ffprobePath = manualPath;
}

function getFFprobePath() {
  return ffprobePath;
}

function findFFprobe() {
  const resourcesPath = process.resourcesPath || __dirname;
  const binaryName = process.platform === 'win32' ? 'ffprobe.exe' : 'ffprobe';
  const archFolders = [`${process.platform}-${process.arch}`];

  if (process.platform === 'darwin') {
    archFolders.push(process.arch === 'arm64' ? 'darwin-x64' : 'darwin-arm64');
  } else if (process.platform === 'win32') {
    archFolders.push(process.arch === 'ia32' ? 'win32-x64' : 'win32-ia32');
  }

  const candidatePaths = [];

  if (ffprobePath) {
    candidatePaths.push(ffprobePath);
    if (ffprobePath.includes('app.asar')) {
      candidatePaths.push(ffprobePath.replace('app.asar', 'app.asar.unpacked'));
    }
  }

  candidatePaths.push(
    path.join(resourcesPath, binaryName),
    path.join(resourcesPath, '..', binaryName),
    path.join(__dirname, '..', '..', binaryName)
  );

  archFolders.forEach((folder) => {
    candidatePaths.push(
      path.join(resourcesPath, 'app.asar.unpacked', 'node_modules', '@ffprobe-installer', folder, binaryName),
      path.join(__dirname, '..', '..', 'app.asar.unpacked', 'node_modules', '@ffprobe-installer', folder, binaryName),
      path.join(__dirname, '..', '..', 'node_modules', '@ffprobe-installer', folder, binaryName)
    );
  });

  if (process.platform === 'darwin') {
    candidatePaths.push('/opt/homebrew/bin/ffprobe', '/usr/local/bin/ffprobe');
  } else if (process.platform === 'linux') {
    candidatePaths.push('/usr/bin/ffprobe', '/usr/local/bin/ffprobe');
  } else if (process.platform === 'win32') {
    const exeDir = path.dirname(process.execPath);
    candidatePaths.push(path.join(exeDir, 'ffprobe.exe'));
  }

  for (const candidate of candidatePaths) {
    if (!candidate || !fs.existsSync(candidate)) {
      continue;
    }

    try {
      if (process.platform !== 'win32') {
        fs.accessSync(candidate, fs.constants.X_OK);
      }
      return candidate;
    } catch (accessError) {
      continue;
    }
  }

  return 'ffprobe';
}

module.exports = {
  initializeFFprobePath,
  setFFprobePath,
  getFFprobePath,
  findFFprobe
};
